<?php
/**
 * OpenApp Payment Gateway for WooCommerce.
 *
 * @package OpenApp_Payment_Gateway
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Load helper class
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/class-openappgw-custom-product.php';

// Load traits
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/traits/trait-hmac-security.php';
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/traits/trait-rest-api-routes.php';
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/traits/trait-cart-storage.php';
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/traits/trait-qr-code.php';
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/traits/trait-order-handling.php';
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/traits/trait-login-integration.php';
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/traits/trait-rest-api-handlers.php';
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/traits/trait-return-management.php';
require_once OPENAPPGW_PLUGIN_DIR_PATH . 'inc/traits/trait-helpers.php';

/**
 * Class OPENAPPGW_OpenApp_Gateway
 *
 * Main payment gateway class extending WC_Payment_Gateway.
 * Uses traits to organize functionality into logical groups.
 */
class OPENAPPGW_OpenApp_Gateway extends WC_Payment_Gateway {

    // Include all traits
    use OPENAPPGW_Trait_HMAC_Security;
    use OPENAPPGW_Trait_REST_API_Routes;
    use OPENAPPGW_Trait_Cart_Storage;
    use OPENAPPGW_Trait_QR_Code;
    use OPENAPPGW_Trait_Order_Handling;
    use OPENAPPGW_Trait_Login_Integration;
    use OPENAPPGW_Trait_REST_API_Handlers;
    use OPENAPPGW_Trait_Return_Management;
    use OPENAPPGW_Trait_Helpers;

    /**
     * API key for OpenApp authentication.
     *
     * @var string
     */
    protected $api_key;

    /**
     * Secret key for HMAC signature generation.
     *
     * @var string
     */
    protected $secret;

    /**
     * OpenApp merchant ID.
     *
     * @var string
     */
    protected $merchant_id;

    /**
     * OpenApp integration profile ID.
     *
     * @var string
     */
    protected $profile_id;

    /**
     * OpenApp API base URL.
     *
     * @var string
     */
    protected $open_app_url;

    /**
     * Payment method title displayed to customers.
     *
     * @var string
     */
    protected $payment_method_title;

    /**
     * Flag to prevent duplicate basket change triggers.
     *
     * @var bool
     */
    private $basket_change_triggered = false;

    /**
     * Cart ID being processed.
     *
     * @var string|null
     */
    private $cart_id_to_process = null;

    /**
     * Supported country code.
     *
     * @var string
     */
    private $supported_country = 'PL';

    /**
     * Available shipping methods mapping.
     *
     * @var array
     */
    private $shipping_methods;

    /**
     * Flag to prevent duplicate QR code rendering in block checkout.
     *
     * @var bool
     */
    private $block_qr_checkout_rendered = false;

    /**
     * Flag to prevent duplicate QR code rendering in block cart.
     *
     * @var bool
     */
    private $block_qr_cart_rendered = false;

    /**
     * Constructor for the gateway.
     */
    public function __construct() {
        $this->id = 'openapp';
        $this->icon = '';
        $this->has_fields = false;
        $this->method_title = __('OpenApp Gateway', 'openapp-gateway-for-woocommerce');
        $this->method_description = __('Convenient online payments through a mobile app', 'openapp-gateway-for-woocommerce');
        $this->payment_method_title = 'OpenApp';
        // other fields
        $this->api_key = $this->get_option('api_key');
        $this->secret = $this->get_option('secret');
        $this->merchant_id = $this->get_option('merchant_id');
        $this->profile_id = $this->get_option('profile_id');
        $this->open_app_url = $this->get_option('open_app_url');

        // Gateways can support subscriptions, refunds, saved payment methods.
        $this->supports = array(
            'products'
        );

        $this->shipping_methods = array(
            ''                      => __('Disabled','openapp-gateway-for-woocommerce'),
            'INPOST_APM'            => __('InPost Paczkomat','openapp-gateway-for-woocommerce'),
            'ORLEN_APM'             => __('Delivery with Orlen Paczka','openapp-gateway-for-woocommerce'),
            'POCZTA_POLSKA_APM'     => __('Delivery with Poczta Polska pickup (machines, pickup points)','openapp-gateway-for-woocommerce'),
            'DHL_PICKUP'            => __('Pickup from DHL pickup point','openapp-gateway-for-woocommerce'),
            'DPD_PICKUP'            => __('Pickup from DPD pickup point','openapp-gateway-for-woocommerce'),
            'INSTORE_PICKUP'        => __('Self-pickup by customer in your store','openapp-gateway-for-woocommerce'),
            'INPOST_COURIER'        => __('Delivery with InPost courier','openapp-gateway-for-woocommerce'),
            'DHL_COURIER'           => __('Delivery using DHL courier','openapp-gateway-for-woocommerce'),
            'DPD_COURIER'           => __('Delivery with DPD courier','openapp-gateway-for-woocommerce'),
            'UPS_COURIER'           => __('Delivery with UPS courier','openapp-gateway-for-woocommerce'),
            'FEDEX_COURIER'         => __('Delivery with FEDEX courier','openapp-gateway-for-woocommerce'),
            'GLS_COURIER'           => __('Delivery with GLS courier','openapp-gateway-for-woocommerce'),
            'POCZTEX_COURIER'       => __('Delivery with POCZTEX courier','openapp-gateway-for-woocommerce'),
            'GEIS_COURIER'          => __('Delivery with GEIS courier','openapp-gateway-for-woocommerce'),
            'ELECTRONIC'            => __('Electronic delivery, eg. license, cinema tickets','openapp-gateway-for-woocommerce')
        );



        $this->init_form_fields();

        // Load the settings.
        $this->init_settings();
        $this->title = $this->get_option( 'title' );
        $this->description = $this->get_option( 'description' );

        add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );

        add_filter('woocommerce_shipping_instance_form_fields_flat_rate', array($this,'add_special_key_field_to_methods'));
        add_filter('woocommerce_shipping_instance_form_fields_free_shipping', array($this,'add_special_key_field_to_methods'));
        add_filter('woocommerce_shipping_instance_form_fields_local_pickup', array($this,'add_special_key_field_to_methods'));

        // integration for: https://pl.wordpress.org/plugins/inpost-paczkomaty/
        add_filter( 'woocommerce_shipping_instance_form_fields_inpost_paczkomaty',array($this,'add_special_key_field_to_methods'));

        // integration for: https://pl.wordpress.org/plugins/flexible-shipping/
        add_filter( 'woocommerce_shipping_instance_form_fields_flexible_shipping_single',array($this,'add_special_key_field_to_methods'));


        // @TODO - maybe move to other place (?)
        add_action('oa_update_cart_in_db', array($this, 'store_cart_in_db'), 10, 1);

        add_action('shutdown', array($this, 'on_shutdown'), 21);
    }


    /**
     * Initialize the gateway functionality.
     * Registers hooks and actions when the gateway is enabled.
     */
    public function init() {

        // Check if the payment gateway is enabled
        if ( ! $this->is_available() ) {
            return;
        }

        // If the 'q' parameter is set (calling static file), return immediately
        $callingStaticFile = isset($_GET['q']) ? sanitize_text_field($_GET['q']) : null;

        if (!empty($callingStaticFile)) {
            return;
        }

        // If the request is a WP cron job, return immediately
        if (defined('DOING_CRON') && DOING_CRON) {
            return;
        }

        // actions for wp-admin
        $this->initAdminActions();

        if ( is_admin() ) {
            return;
        }

        // @TODO - move to other place (?)
        add_action('wp_enqueue_scripts', array($this,'oa_plugin_enqueue_scripts'));

        $this->registerAPIRoutes();
        $this->registerCartStorage();
        $this->registerOaOrder();

        // Check if the OALogin is enabled
        if ('yes' === $this->get_option('oalogin_enabled', 'no')) {
            $this->registerOaLogin();
        }

    }

    /**
     * Initialize admin-specific actions.
     */
    public function initAdminActions(){
        add_action('woocommerce_order_status_changed', array($this, 'oa_status_changed'), 10, 4);
        add_action('current_screen', array($this,'sse_support_script') );
        add_action('wp_ajax_test_sse_support', array($this,'test_sse_support_callback'));
        add_action('wp_ajax_sse_save_test_result', array($this,'handle_sse_save_test_result'));

        // order-returns
        add_action('add_meta_boxes', array($this, 'add_order_returns_meta_box'));
        add_action('wp_ajax_openapp_handle_return_action', array($this, 'handle_return_action_ajax'));

        add_filter('rest_pre_serve_request', array($this, 'prevent_caching'), 10, 4);

    }

    /**
     * ===========================================================================
     * WooCommerce Payment Gateway Settings
     * ===========================================================================
     */

    /**
     * Initialize form fields for the gateway settings page.
     */
    public function init_form_fields(){
        $this->form_fields = array(
            'enabled' => array(
                'title'       => __('Enable/Disable','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable OpenApp Gateway','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => '',
                'default'     => 'no'
            ),
            'title' => array(
                'title'       => __('Title','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('This controls the title which the user sees during checkout.','openapp-gateway-for-woocommerce'),
                'default'     => __('OpenApp Payment','openapp-gateway-for-woocommerce'),
                'desc_tip'    => true,
            ),
            'description' => array(
                'title'       => __('Description','openapp-gateway-for-woocommerce'),
                'type'        => 'textarea',
                'description' => __('This controls the description which the user sees during checkout.','openapp-gateway-for-woocommerce'),
                'default'     => __('Pay with OpenApp payment gateway.','openapp-gateway-for-woocommerce'),
            ),
            //...other settings
            'api_key' => array(
                'title'       => __('API Key','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('Enter your OpenApp API key here.','openapp-gateway-for-woocommerce'),
                'default'     => '',
            ),
            'secret' => array(
                'title'       => __('Secret','openapp-gateway-for-woocommerce'),
                'type'        => 'password',
                'description' => __('Enter your OpenApp secret here.','openapp-gateway-for-woocommerce'),
                'default'     => '',
            ),
            'merchant_id' => array(
                'title'       => __('Merchant ID','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('Enter your OpenApp merchant ID here.','openapp-gateway-for-woocommerce'),
                'default'     => '',
            ),
            'profile_id' => array(
                'title'       => __('Profile ID','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('Enter your OpenApp profile ID here.','openapp-gateway-for-woocommerce'),
                'default'     => '',
            ),
            'open_app_url' => array(
                'title'       => __('API Base Url','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('Enter OpenApp API base url here.','openapp-gateway-for-woocommerce'),
                'default'     => 'https://api.uat.open-pay.com/merchant',
            ),
            'interval_time' => array(
                'title'       => __('Interval Time','openapp-gateway-for-woocommerce'),
                'type'        => 'select',
                'description' => __('Select the interval time for order redirection checking (AJAX polling). A smaller interval provides a faster redirection experience but may increase server load.','openapp-gateway-for-woocommerce'),
                'default'     => '8500',
                'options'     => array(
                    '2000'  => __('2 seconds','openapp-gateway-for-woocommerce'),
                    '5000'  => __('5 seconds','openapp-gateway-for-woocommerce'),
                    '8500'  => __('8.5 seconds','openapp-gateway-for-woocommerce')
                ),
            ),
            'sse_enabled' => array(
                'title'       => __('Server-Sent Events (SSE)','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable SSE for Order Redirection','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => sprintf(
                    __('When enabled, the plugin utilizes Server-Sent Events (SSE) to manage order check and thank you page redirection. This method is more efficient than traditional AJAX polling. Before enabling SSE, you can check your server\'s compatibility by clicking on the link: %s', 'openapp-gateway-for-woocommerce'),
                    '<a href="#" id="test-sse-button">' . __('Test SSE Support', 'openapp-gateway-for-woocommerce') . '</a><span id="ct-sse-test-result" style="margin-left:5px;"></span>'
                ),
                'default'     => 'no',
            ),
            'order_status' => array(
                'title'       => __('Order Status','openapp-gateway-for-woocommerce'),
                'type'        => 'select',
                'description' => __('Select the default order status for new orders, example: Processing','openapp-gateway-for-woocommerce'),
                'default'     => 'wc-processing',
                'options'     => wc_get_order_statuses()
            ),
            'basket_sync' => array(
                'title'       => __('Cart synchronization','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable cart synchronization','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => __('Activating this option will synchronize every change made to the WooCommerce shopping cart with the OpenApp mobile app in real-time.','openapp-gateway-for-woocommerce'),
                'default'     => 'no',
            ),
            'oalogin_enabled' => array(
                'title'       => __('Enable/Disable OALogin','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable OALogin','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => __('If checked, the OALogin functionality will be enabled. Uncheck this option to disable OALogin.','openapp-gateway-for-woocommerce'),
                'default'     => 'yes',
            ),
            'validation_enabled' => array(
                'title'       => __('Enable/Disable Validation','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable request validation','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => __('If checked, the plugin will validate OpenApp incoming requests. Disable this option only for testing purposes.','openapp-gateway-for-woocommerce'),
                'default'     => 'yes',
            ),
            'debug' => array(
                'title'       => __('Debug','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable logging','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => __('If checked, the plugin will log debug information to the PHP error log.','openapp-gateway-for-woocommerce'),
                'default'     => 'no',
            ),
        );
    }

    /**
     * Payment fields displayed on the checkout page.
     */
    public function payment_fields() {}

    /**
     * Process the payment and return the result.
     *
     * @param int $order_id Order ID.
     * @return array Result array with 'result' and 'redirect' keys.
     */
    public function process_payment( $order_id ) {
        global $woocommerce;

        $order = wc_get_order( $order_id );
        $woocommerce->cart->empty_cart();
        return array(
            'result'   => 'success',
            'redirect' => $this->get_return_url( $order )
        );
    }

    /**
     * Check if the gateway is available for use.
     *
     * @return bool True if available, false otherwise.
     */
    public function is_available() {
        $is_available = ('yes' === $this->get_option('enabled', 'no'));
        return $is_available;
    }

    /**
     * ===========================================================================
     * SSE Support Methods
     * ===========================================================================
     */

    /**
     * Check if SSE is enabled and supported.
     *
     * @return bool True if SSE is enabled and supported.
     */
    public static function is_sse_enabled() {
        // Retrieve the SSE support status from the transient
        $sseSupportStatus = get_transient('openappgw_sse_supported');

        // Check if the server supports SSE
        $serverSupportsSSE = ($sseSupportStatus !== false && $sseSupportStatus === "1");

        // Retrieve the WooCommerce option and check if SSE is enabled
        $options = get_option('woocommerce_openapp_settings');
        $sseEnabled = isset($options['sse_enabled']) && $options['sse_enabled'] === 'yes';

        // Return true only if both server supports SSE and the SSE option is enabled
        return $serverSupportsSSE && $sseEnabled;
    }

    /**
     * Add SSE support test script on settings page.
     */
    public function sse_support_script() {
        $page = isset($_GET['page']) ? sanitize_text_field($_GET['page']) : '';
        $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        $section = isset($_GET['section']) ? sanitize_text_field($_GET['section']) : '';

        if (!empty($page) && !empty($tab) && !empty($section) &&
            $page === 'wc-settings' &&
            $tab === 'checkout' &&
            $section === 'openapp') {
            add_action( 'admin_enqueue_scripts', array($this,'enqueue_sse_test_script') );
        }
    }

    /**
     * Enqueue the SSE test script.
     */
    public function enqueue_sse_test_script() {
        wp_enqueue_script( 'openappgw-admin-sse-test', OPENAPPGW_PLUGIN_DIR_URL . 'assets/js/sse-test.js', array(), OPENAPPGW_WOOCOMMERCE_GATEWAY, true );

        wp_localize_script('openappgw-admin-sse-test', 'sseTestParams', array(
            'ajaxUrlTest' => admin_url('admin-ajax.php?action=test_sse_support&security=' . wp_create_nonce('openappgw_sse_support_nonce')),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('openappgw_sse_support_nonce')
        ));

    }

    /**
     * AJAX callback for testing SSE support.
     */
    public function test_sse_support_callback() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this feature.'));
        }

        check_ajax_referer('openappgw_sse_support_nonce', 'security');

        // Set the headers for SSE
        header('Content-Type: text/event-stream');
        header('Cache-Control: no-cache');
        header('Connection: keep-alive');
        header('X-Accel-Buffering: no');

        // Run a loop to send messages periodically
        for ($i = 0; $i < 3; $i++) {
            echo "event: test\n";
            echo "data: " . wp_json_encode(array('message' => 'SSE test message')) . "\n\n";
            // Flush the output buffer to the client
            if (ob_get_level() > 0) ob_flush();
            flush();
            // Sleep for a second to simulate a delay between messages
            sleep(1);
        }

        // Close the connection after the test
        echo "event: close\n";
        echo "data: " . wp_json_encode(array('message' => 'Test complete')) . "\n\n";
        if (ob_get_level() > 0) ob_flush();
        flush();

        // Terminate the script without displaying the shutdown HTML
        exit(0);
    }

    /**
     * AJAX handler to save SSE test result.
     */
    public function handle_sse_save_test_result() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this feature.'));
        }

        check_ajax_referer('openappgw_sse_support_nonce', 'security');

        // Directly compare with the string 'true' or 'false'
        // Compare with the string 'true'
        $sseSupported = (isset($_POST['sseSupported']) && $_POST['sseSupported'] === 'true');
        // Cast the boolean to an integer (true to 1, false to 0)
        $sseSupportedInt = (int) $sseSupported;

        set_transient('openappgw_sse_supported', $sseSupportedInt, 0);

        wp_send_json_success(__('SSE support status updated.','openapp-gateway-for-woocommerce'));
    }

    /**
     * Prevent caching of OpenApp REST API responses.
     *
     * @param bool            $served  Whether the request has already been served.
     * @param WP_HTTP_Response $result  Result to send to the client.
     * @param WP_REST_Request  $request Request used to generate the response.
     * @param WP_REST_Server   $server  Server instance.
     * @return bool Whether the request has been served.
     */
    public function prevent_caching($served, $result, $request, $server) {
        $route = $request->get_route();

        if (strpos($route, '/openapp/') !== false) {
            header("Cache-Control: no-cache, no-store, must-revalidate");
            header("Pragma: no-cache");
            header("Expires: 0");
        }

        return $served;
    }

    /**
     * ===========================================================================
     * Shipping Methods
     * ===========================================================================
     */

    /**
     * Convert a decimal value to cents (minor currency unit).
     *
     * @param mixed $value The value to convert.
     * @return int The value in cents.
     */
    private function convertToCents($value) {
        // Handle null/empty values
        if ($value === null || $value === '') {
            return 0;
        }

        // Replace comma with dot for decimal separator (only comma/dot for cents)
        $normalized = str_replace(',', '.', (string) $value);

        // Convert to float, multiply by 100, and round with explicit mode
        return (int) round((float) $normalized * 100, 0, PHP_ROUND_HALF_UP);
    }

    /**
     * Rebuild cart contents with product data from stored cart data.
     *
     * @param array $cart_data The stored cart data.
     * @return array The rebuilt cart contents with product data objects.
     */
    private function rebuild_cart_contents_with_data($cart_data) {
        // Extract simplified cart contents and cart contents data from $cart_data
        $simplified_cart_contents = $cart_data['cart_contents'];
        $cart_contents_data = $cart_data['cart_contents_data'];

        $rebuilt_cart_contents = [];

        foreach ($simplified_cart_contents as $cart_item_key => $simplified_cart_item) {
            // If there is corresponding 'data' for this cart item, add it back
            if (isset($cart_contents_data[$cart_item_key])) {
                $customProduct = new OPENAPPGW_CustomProduct($cart_contents_data[$cart_item_key]);
                $simplified_cart_item['data'] = $customProduct;
            }

            // Add the rebuilt cart item to the array
            $rebuilt_cart_contents[$cart_item_key] = $simplified_cart_item;
        }

        return $rebuilt_cart_contents;
    }

    /**
     * Calculate shipping cost for cart using a specific shipping method.
     *
     * @deprecated No longer used. Shipping calculation is now handled by
     *             get_available_shipping_methods() using WC's native pipeline.
     *
     * @param array  $cart_contents_record The cart contents record from database.
     * @param object $method               The shipping method instance.
     * @param int    $basket_value         The basket value in cents.
     * @return int|null The shipping cost in cents, or null on error.
     */
    public function calculate_shipping_cost_for_cart($cart_contents_record, $method, $basket_value) {
        _deprecated_function(__METHOD__, '1.0.0', 'get_available_shipping_methods');

        try {
            $woocommerce = WC();

            if ( ! isset($woocommerce->session)) {
                return null;
            }

            if ( null === $woocommerce->cart ) {
                $woocommerce->cart = new WC_Cart();
            }

            $woocommerce->cart->empty_cart();

            $cart_data = $this->rebuild_cart_contents_with_data($cart_contents_record);

            foreach ($cart_data as $item) {
                $_product = $this->get_product_object($item);
                if(is_null($_product)){
                    continue;
                }

                $product_id = $_product->get_id();
                $quantity = $item['quantity'];
                $woocommerce->cart->add_to_cart($product_id, $quantity);
            }

            $package = array(
                'destination' => array(
                    'country' => $this->supported_country,
                ),
                'contents' => $cart_data,
                'applied_coupon' => [],
                'contents_cost' => $basket_value
            );

            $method->calculate_shipping($package);

            $rates = $method->rates;
            $total_shipping_cost = 0;

            foreach ($rates as $rate_id => $rate) {
                $rate_cost = $rate->get_cost();
                $total_shipping_cost += $rate_cost;
            }

            return $this->convertToCents($total_shipping_cost);
        } catch (Exception $e) {
            // Handle exception
            $this->log_debug('Error occurred: ' . $e->getMessage());
            // You can return a specific value or even re-throw the exception
            return null; // Or any other appropriate response
        }
    }

    /**
     * Get available shipping methods for a country with their costs.
     *
     * Uses WooCommerce's native shipping calculation pipeline which:
     * - Respects all is_available() checks (min_amount, coupons, rules)
     * - Honors all filters (woocommerce_package_rates, etc.)
     * - Handles zone matching automatically
     *
     * @param string $country_code        The country code.
     * @param int    $basket_value        The basket value in cents (unused, kept for backward compatibility).
     * @param array  $cart_contents_record The cart contents record for cost calculation.
     * @return array Array of available shipping methods with keys and costs.
     */
    public function get_available_shipping_methods($country_code, $basket_value, $cart_contents_record = array()) {
        $cheapest_methods = array();

        if (!class_exists('WC_Shipping_Zones')) {
            return array();
        }

        try {
            $woocommerce = WC();

            if (!isset($woocommerce->session)) {
                return array();
            }

            // Initialize cart if needed
            if (null === $woocommerce->cart) {
                $woocommerce->cart = new WC_Cart();
            }

            // Clear and rebuild cart with real products
            $woocommerce->cart->empty_cart();
            $cart_data = $this->rebuild_cart_contents_with_data($cart_contents_record);

            foreach ($cart_data as $item) {
                $_product = $this->get_product_object($item);
                if (is_null($_product)) {
                    continue;
                }

                // Handle variations properly - pass variation_id and attributes
                $product_id = isset($item['product_id']) ? (int) $item['product_id'] : $_product->get_id();
                $variation_id = isset($item['variation_id']) ? (int) $item['variation_id'] : 0;
                $variation = isset($item['variation']) ? $item['variation'] : array();

                $woocommerce->cart->add_to_cart($product_id, $item['quantity'], $variation_id, $variation);
            }

            // Restore coupons if available (needed for coupon-based free shipping)
            if (!empty($cart_contents_record['coupon_data'])) {
                foreach ($cart_contents_record['coupon_data'] as $coupon) {
                    if (!empty($coupon['code'])) {
                        $woocommerce->cart->apply_coupon($coupon['code']);
                    }
                }
            }

            // Build proper package from WC cart
            $package = array(
                'destination' => array(
                    'country'   => $country_code,
                    'state'     => '',
                    'postcode'  => '',
                    'city'      => '',
                    'address'   => '',
                    'address_2' => '',
                ),
                'contents'        => $woocommerce->cart->get_cart(),
                'contents_cost'   => $woocommerce->cart->get_cart_contents_total(),
                'applied_coupons' => $woocommerce->cart->get_applied_coupons(),
                'user'            => array('ID' => get_current_user_id()),
            );

            // Use WC's full shipping calculation pipeline
            $woocommerce->shipping()->calculate_shipping(array($package));

            // Get calculated rates
            $packages = $woocommerce->shipping()->get_packages();
            $rates = isset($packages[0]['rates']) ? $packages[0]['rates'] : array();

            // Map rates to oa_shipping_mapping
            $shipping_method_classes = $woocommerce->shipping()->get_shipping_methods();

            foreach ($rates as $rate) {
                $method_id = $rate->get_method_id();
                $instance_id = $rate->get_instance_id();

                // Skip if we can't identify the method
                if (!isset($shipping_method_classes[$method_id])) {
                    continue;
                }

                // Get the shipping method instance to read oa_shipping_mapping
                $method_class = $shipping_method_classes[$method_id];
                $method = new $method_class($instance_id);
                $oa_shipping_method_key = $method->get_option('oa_shipping_mapping');

                if (empty($oa_shipping_method_key)) {
                    continue;
                }

                $cost = $this->convertToCents($rate->get_cost());

                // Keep only the cheapest rate per oa_shipping_mapping
                if (!isset($cheapest_methods[$oa_shipping_method_key]) ||
                    $cost < $cheapest_methods[$oa_shipping_method_key]['cost']) {
                    $cheapest_methods[$oa_shipping_method_key] = array(
                        'key'  => $oa_shipping_method_key,
                        'cost' => (int) $cost
                    );
                }
            }

        } catch (Exception $e) {
            $this->log_debug('Error calculating shipping: ' . $e->getMessage());
            return array();
        }

        return array_values($cheapest_methods);
    }

    /**
     * Add OpenApp mapping field to shipping method settings.
     *
     * @param array $fields The existing shipping method fields.
     * @return array The modified fields with OpenApp mapping added.
     */
    public function add_special_key_field_to_methods($fields) {

        $fields['oa_shipping_mapping'] = array(
            'title'         => __('OpenApp mapping', 'openapp-gateway-for-woocommerce'),
            'type'          => 'select',
            'description'   => __('Select shipping method used in OpenApp mobile app', 'openapp-gateway-for-woocommerce'),
            'options'       => $this->shipping_methods,
            'default'       => '',
        );

        return $fields;
    }

    /**
     * ===========================================================================
     * Hook Registration Methods
     * ===========================================================================
     */

    /**
     * Register REST API routes.
     */
    private function registerAPIRoutes(){
        add_filter( 'woocommerce_is_rest_api_request',  array($this,'exclude_specific_endpoint_from_rest') );

        add_action('rest_api_init', array($this, 'register_openapp_routes_external'));
        add_action('rest_api_init', array($this, 'register_openapp_routes_internal'));
    }

    /**
     * Register cart storage hooks.
     */
    private function registerCartStorage(){
        add_action('woocommerce_cart_loaded_from_session', array($this,'store_previous_cart_hash_in_session'), 99, 1);
        add_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

        // Migrate cart session when guest logs in (WC 3.6+)
        // Set flag on login (wp_login fires BEFORE WC merges carts)
        add_action('wp_login', array($this, 'refresh_cart_after_login'), 20, 2);

        // Persist merged cart AFTER WC has loaded/merged from session (priority 100 = after store_previous_cart_hash_in_session)
        add_action('woocommerce_cart_loaded_from_session', array($this, 'maybe_refresh_cart_after_login'), 100, 1);
    }

    /**
     * Register order-related hooks and QR code display.
     */
    private function registerOaOrder(){
        add_action('woocommerce_after_cart_table', array($this, 'openapp_qr_order_as_action'), 1);
        add_action('woocommerce_review_order_before_payment', array($this, 'openapp_qr_order_as_action'), 1);
        add_action('wp', array($this,'register_oa_order_shortcode'));
        add_action('woocommerce_thankyou_' . $this->id, array($this, 'oa_reset_order_key'), 2);
        add_filter('render_block', array($this, 'inject_qr_into_block_cart_checkout'), 20, 2);
    }

    /**
     * Register OpenApp login-related hooks.
     */
    private function registerOaLogin(){
        add_action('wp_loaded', array($this, 'set_guest_session'));
        add_action('init', array($this,'oa_custom_login'));

        add_action('woocommerce_login_form_end', array($this,'openapp_qr_login_as_action'));
        add_action('wp', array($this,'register_oa_login_shortcode'));
    }

    /**
     * After a user logs in, set flag to refresh cart on next cart load.
     * We don't persist here because wp_login fires BEFORE WooCommerce merges carts.
     *
     * @param string  $user_login Username.
     * @param WP_User $user       User object.
     */
    public function refresh_cart_after_login($user_login, $user) {

        if (!function_exists('WC')) {
            return;
        }

        // Ensure session exists
        $this->ensure_wc_session_loaded();

        if (!isset(WC()->session)) {
            return;
        }

        // Don't persist here - WC hasn't merged carts yet (wp_login fires before woocommerce_guest_session_to_user_id)
        // Set flag so we persist on next cart load when WC has merged the data
        WC()->session->set('oa_needs_cart_refresh', true);
        $this->log_debug('refresh_cart_after_login: set oa_needs_cart_refresh flag for user ' . $user->ID);
    }

    /**
     * Persist cart to OA table after WC has loaded/merged cart from session.
     * This fires AFTER woocommerce_guest_session_to_user_id, so cart data is merged.
     *
     * @param WC_Cart $cart The cart object.
     */
    public function maybe_refresh_cart_after_login($cart) {
        if (!isset(WC()->session)) {
            return;
        }

        if (WC()->session->get('oa_needs_cart_refresh')) {
            WC()->session->set('oa_needs_cart_refresh', false);
            $this->store_cart_in_db(true);
        }
    }



}
