<?php
/**
 * Order Handling trait for OpenApp Gateway.
 *
 * @package OpenApp_Payment_Gateway
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Trait OPENAPPGW_Trait_Order_Handling
 *
 * Handles order creation, status synchronization, and redirection.
 */
trait OPENAPPGW_Trait_Order_Handling {

    public function oa_check_order_redirection(WP_REST_Request $request)
    {
        $cart_id = sanitize_text_field($request->get_param('cart_id'));
        $useSSE = sanitize_text_field($request->get_param('use_sse'));

        // Verify cart ownership (prevents enumeration attacks)
        $cart_ownership = $this->verify_cart_ownership($cart_id);
        if (is_wp_error($cart_ownership)) {
            return $this->create_secure_response(['redirect' => false]);
        }

        $thank_you_url = $this->get_thank_you_url($cart_id);

        if($useSSE){
            header('Content-Type: text/event-stream');
            header('Cache-Control: no-cache');
            header('Connection: keep-alive');
            header('X-Accel-Buffering: no');

            // set_time_limit(0);
            ignore_user_abort(true);

            $lastEventId = 0;

            while (true) {
                while (ob_get_level() > 0) {
                    ob_end_flush();
                }
                flush();

                $thank_you_url = $this->get_thank_you_url($cart_id);
                $lastEventId++;

                $this->openappgw_custom_log($lastEventId . ' ' . $thank_you_url, 'sse');

                if ($thank_you_url) {
                    echo "id: " . esc_html($lastEventId) . "\n";
                    echo "event: orderUpdate\n";
                    echo 'data: ' . wp_json_encode(['redirect' => true, 'url' => $thank_you_url]) . "\n\n";
                } else {
                    echo "id: " . esc_html($lastEventId) . "\n";
                    echo "event: orderUpdate\n";
                    echo 'data: ' . wp_json_encode(['redirect' => false]) . "\n\n";
                }

                flush();

                // Break the loop if a URL is found
                if ($thank_you_url) {
                    break;
                }

                sleep(3);
            }

        } else {
            // Non-SSE mode: return with secure headers
            if ($thank_you_url) {
                return $this->create_secure_response(['redirect' => true, 'url' => $thank_you_url]);
            }
            return $this->create_secure_response(['redirect' => false]);
        }
    }

    public function get_thank_you_url($cart_id) {
        global $wpdb;

        $result = $wpdb->get_row(
            $wpdb->prepare("SELECT order_key FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $cart_id)
        );

        if(!empty($result) && $result->order_key !== NULL) {
            // order_key column contains order_number
            $order_id = $result->order_key;
            if ($order_id) {
                $order_key = null;
                $order = wc_get_order($order_id);
                if ($order && !is_wp_error($order)) {
                    $order_key = $order->get_order_key();
                }

                if(!is_null($order_key)){
                    $checkout_url = trailingslashit(wc_get_checkout_url()); // ensures that URL ends with a slash
                    $thank_you_url = add_query_arg(['key' => $order_key, 'order-received' => $order_id], $checkout_url . 'order-received/');

                    // Reset order key
                    // $this->oa_reset_order_key($order_id, false);

                    // Validate the URL
                    if (filter_var($thank_you_url, FILTER_VALIDATE_URL) !== false) {
                        // The URL is valid, proceed with the redirect
                        return $thank_you_url;
                    }
                }
            }
        }

        return false;
    }

    public function oa_reset_order_key($order_id, $assignOrder = true) {
        global $wpdb;

        $this->log_debug("reset order key");

        // Set the order_key back to NULL
        $wpdb->update(
            $wpdb->prefix . "oa_woocommerce_persistent_cart",
            ['order_key' => NULL],
            ['order_key' => $order_id]  // Where order_key matches
        );

        if($assignOrder){
            // assign guest order to customer
            $order = wc_get_order($order_id);
            $user_email = $order->get_billing_email();
            $user = get_user_by('email', $user_email);

            if (!$user) {
                $random_password = wp_generate_password(16, true);
                $user_id = $this->createWpUser($user_email, $random_password);

                if (is_wp_error($user_id)) {

                }
            } else {
                $user_id = $user->ID;
            }
            $order->set_customer_id($user_id);
            $order->save();


            // Trigger status update in API
            $current_status = $order->get_status();
            $this->oa_status_changed($order_id, $current_status, $current_status, $order);
        }

    }

    public function oa_status_changed($order_id, $from_status, $to_status, $order){
        $endpoint = '/v1/orders/multiFulfillment';
        $method = 'POST';
        $url = $this->open_app_url . $endpoint ;
        $context = 'openapp_basket_fulfillment';


        // Mapping WooCommerce statuses to the ones expected by OpenApp
        $status_mapping = array(
            'completed' => 'SHIPPED',
            'processing' => 'FULFILLED',
            'shipped' => 'SHIPPED',
            'ready-for-pickup' => 'READY_FOR_PICKUP',
            'cancelled' => 'CANCELLED_MERCHANT'
        );

        // Check if the changed status exists in our mapping
        if (!isset($status_mapping[$to_status])) {
            return; // Exit if we don't need to send data for this status
        }
        // Retrieve oaOrderId from the order's post meta
        $oaOrderId = get_post_meta($order_id, 'oaOrderId', true);

        if (!$oaOrderId) {
            return;
        }

        /**
         * get shipping method
         */
        $order = wc_get_order($order_id);
        if ( ! is_a( $order, 'WC_Order' ) ) {
            return;
        }

        $shipments = [];

        // Get the shipping methods from the order
        $shipping_methods = $order->get_shipping_methods();

        foreach ($shipping_methods as $shipping_method) {
            // Construct each shipment based on the shipping method
            $method_id = $shipping_method->get_method_id();
            $shipment = [
                'shipmentId' => $order_id .'_'. $method_id,
                'status' => $status_mapping[$to_status],
                'notes' => '',
                'operator' => $method_id,
            ];

            $shipments[] = $shipment;
        }

        // Prepare the order data for the API request
        $order_data = [
            'oaOrderId' => $oaOrderId,
            'shopOrderId' => (string)$order_id,
            'shipments' => $shipments
        ];

        $body = $this->encodeJsonResponse($order_data);
        $setupData = $this->setupRequest($endpoint, $method, $body, $context);
        $response = $this->executeRequest($url, $setupData, $context);
    }

    private function setupRequest($endpoint, $method, $body, $context) {
        $timestamp = (int) (microtime(true) * 1000);
        $nonce = $timestamp;
        $path = strtoupper($endpoint);

        $bodyHash = hash('sha256', $body, true);
        $bodyHashBase64 = $this->base64Encode($bodyHash);
        $hmacSignature = $this->generateHmacSignature($method, $path, $timestamp, $nonce, $bodyHashBase64, null, null, $context);
        $authHeader = "hmac v1$" . $this->api_key . "$" . $method . "$" . $path . "$" . $timestamp . "$" . $nonce;

        $this->openappgw_custom_log("Request Authorization Header: " . $authHeader, $context);
        $this->openappgw_custom_log("myBody: ". var_export($body, true), $context);

        return [
            'method' => $method,
            'headers' => array(
                'Content-Type' => 'application/json',
                'authorization' => sanitize_text_field($authHeader),
                'x-app-signature' => sanitize_text_field($hmacSignature)
            ),
            'body' => $body
        ];
    }

    private function executeRequest($url, $setupData, $context) {
        $method = $setupData['method'] ?? 'POST';

        $request_args = array(
            'headers' => $setupData['headers'],
            'body' => $setupData['body'],
            'timeout' => 10,
            'method' => $method,
            'blocking' => true
        );

        $this->openappgw_custom_log("=== EXECUTING REQUEST ===", $context);
        $this->openappgw_custom_log("URL: " . $url, $context);
        $this->openappgw_custom_log("Method: " . $method, $context);

        $response = wp_remote_request($url, $request_args);

        $responseBody = wp_remote_retrieve_body($response);
        $http_code = wp_remote_retrieve_response_code($response);

        // Logging response
        $this->openappgw_custom_log("RESPONSE:", $context);
        $this->openappgw_custom_log("HTTP Code: " . $http_code, $context);
        $this->openappgw_custom_log("Response Body: " . print_r($responseBody, true), $context);

        if (in_array($http_code, [200, 204])) {
            $this->openappgw_custom_log("Successful response received", $context);
        } else {
            $this->openappgw_custom_log("Error response received", $context);
        }

        return $response;
    }

}
