<?php
/**
 * QR Code trait for OpenApp Gateway.
 *
 * @package OpenApp_Payment_Gateway
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Trait OPENAPPGW_Trait_QR_Code
 *
 * Handles QR code generation, shortcodes, and block injection.
 */
trait OPENAPPGW_Trait_QR_Code {

    public function openapp_qr_order_as_action(){
        echo wp_kses_post($this->openapp_qr_order());
    }

    public function openapp_qr_login_as_action() {
        echo wp_kses_post($this->openapp_qr_login());
    }

    public function register_oa_order_shortcode() {
        $is_internal_api = isset($_SERVER['HTTP_X_WP_INTERNAL']) && sanitize_text_field($_SERVER['HTTP_X_WP_INTERNAL']) == 'true';

        if (is_admin() || (defined('DOING_CRON') && DOING_CRON) || wp_doing_ajax() || (defined('REST_REQUEST') && REST_REQUEST) || $is_internal_api) {
            return;
        }

        // enable only on page OR post
        if(is_page() || is_single() || is_archive()){
            $sanitized_uri = filter_var($_SERVER['REQUEST_URI'], FILTER_SANITIZE_URL);
            $this->log_debug('Shortcode triggered by ' . $sanitized_uri);

            add_shortcode('openapp_qr_order', array($this, 'openapp_qr_order'));
        }
    }

    public function register_oa_login_shortcode(){
        $is_internal_api = isset($_SERVER['HTTP_X_WP_INTERNAL']) && sanitize_text_field($_SERVER['HTTP_X_WP_INTERNAL']) == 'true';

        if (is_admin() || (defined('DOING_CRON') && DOING_CRON) || wp_doing_ajax() || (defined('REST_REQUEST') && REST_REQUEST) || $is_internal_api) {
            return;
        }

        // enable only on page OR post
        if(is_page() || is_single() || is_archive()){
            $sanitized_uri = filter_var($_SERVER['REQUEST_URI'], FILTER_SANITIZE_URL);
            $this->log_debug('Shortcode triggered by ' . $sanitized_uri);

            add_shortcode('openapp_qr_login', array($this, 'openapp_qr_login'));
        }
    }

    public function oa_plugin_enqueue_scripts() {
        // Register the script
        wp_register_script('openappgw-js', 'https://static.prd.open-pay.com/dist/barcode-library/openapp.min.1.1.0.js', array('jquery'), '1.1.0', false);
    }

    public function inject_qr_into_block_cart_checkout($block_content, $block){
        if (is_admin()) {
            return $block_content;
        }

        if (!isset($block['blockName'])) {
            return $block_content;
        }

        $block_name = $block['blockName'];

        if (!$this->is_available()) {
            return $block_content;
        }

        // Define target blocks
        $checkout_summary_blocks = array(
            'woocommerce/checkout-order-summary',
            'woocommerce/checkout-order-summary-block',
        );
        $checkout_actions_blocks = array(
            'woocommerce/checkout-actions-block',
            'woocommerce/checkout-place-order-block',
        );
        $cart_totals_blocks = array(
            'woocommerce/cart-totals-block',
            'woocommerce/cart-order-summary-block',
        );

        // Determine block type
        $is_checkout_summary = in_array($block_name, $checkout_summary_blocks, true);
        $is_checkout_actions = in_array($block_name, $checkout_actions_blocks, true);
        $is_cart_totals = in_array($block_name, $cart_totals_blocks, true);

        // Early return if not a target block
        if (!$is_checkout_summary && !$is_checkout_actions && !$is_cart_totals) {
            return $block_content;
        }

        // Early return if already rendered for checkout context
        if (($is_checkout_summary || $is_checkout_actions) && $this->block_qr_checkout_rendered) {
            return $block_content;
        }

        // Early return if already rendered for cart context
        if ($is_cart_totals && $this->block_qr_cart_rendered) {
            return $block_content;
        }

        // NOW call the function (only once per context)
        $qr_markup = $this->openapp_qr_order();
        if (empty($qr_markup)) {
            return $block_content;
        }

        // Append markup and set flag
        if ($is_checkout_summary) {
            $this->block_qr_checkout_rendered = true;
            $block_content .= '<div class="oa-checkout-summary-qr">' . $qr_markup . '</div>';
        } elseif ($is_checkout_actions) {
            $this->block_qr_checkout_rendered = true;
            $block_content .= '<div class="oa-checkout-actions-qr">' . $qr_markup . '</div>';
        } elseif ($is_cart_totals) {
            $this->block_qr_cart_rendered = true;
            $block_content .= '<div class="oa-cart-totals-qr">' . $qr_markup . '</div>';
        }

        return $block_content;
    }

    public function openapp_qr_order($atts = array()){
        $sanitized_uri = filter_var($_SERVER['REQUEST_URI'], FILTER_SANITIZE_URL);
        $this->log_debug('openapp_qr_order triggered at ' . gmdate('Y-m-d H:i:s') . ' by ' . $sanitized_uri);


        // parse the shortcode attributes
        $args = shortcode_atts(array(
            'lang' => sanitize_text_field(get_bloginfo('language')),
        ), $atts, 'openapp_qr_order');

        wp_enqueue_script('openappgw-js');

        global $woocommerce;
        $sessionId = $this->get_session_id();
        $cart_is_empty = $woocommerce->cart->is_empty();

        // Ensure cart is persisted even on empty GET requests (Blocks render path).
        $cart_id = $this->get_cart_id_by_session($sessionId);
        if (empty($cart_id) && 'yes' === $this->get_option('basket_sync', 'no')) {
            $this->store_cart_in_db(true);
            $cart_id = $this->get_cart_id_by_session($sessionId);
        }

        // Enqueue the script (only if it hasn't been enqueued already)
        if(!wp_script_is('openappgw-js-2', 'enqueued')) {
            wp_enqueue_script('openappgw-js-2', OPENAPPGW_PLUGIN_DIR_URL . 'assets/js/openapp-shortcodes-1.js', array('jquery'), OPENAPPGW_WOOCOMMERCE_GATEWAY, true);

            wp_enqueue_style('openapp-css', OPENAPPGW_PLUGIN_DIR_URL . 'assets/css/openapp-shortcode.css', array(), OPENAPPGW_WOOCOMMERCE_GATEWAY);

            // Pass PHP variables to JavaScript
            $localize_data = array(
                'baseUrl' => $this->get_base_url(),
                'cartId' => $cart_id,
                'qrCodeUrl' => esc_url_raw( rest_url( 'openapp/v1/qr_code' ) ),
                'oaRedirectionUrl' => esc_url_raw( rest_url( 'openapp/v1/oa_redirection' ) ),
                'errorTextMessage' => __('OpenApp QR Code. Your cart is empty!','openapp-gateway-for-woocommerce'),
                'cartIsEmpty' => $cart_is_empty,
                'intervalTime' => $this->get_option('interval_time'),
                'sseEnabled' => false
            );

            if($this->is_sse_enabled()){
                $localize_data['sseEnabled'] = true;
            }

            wp_localize_script('openappgw-js-2', 'openappVars', $localize_data);
        }

        ob_start(); // Start capturing output into a buffer
        // Your CSS
        $output = '';

        $output .= '<div class="OpenAppCheckout-loading OpenAppCheckoutOrder" data-letter="C" data-style=""
data-merchantId="" data-integrationProfileId="" data-basketId="" data-basketValue="" data-basketCurrency="" data-uniqueProductsCount=""
data-lang="'. $args['lang'] .'"
></div>';


       if(is_null($sessionId)){
           return "";
       }

        ob_end_clean(); // End capture and discard buffer

        $output = $this->minify_string($output);

        return $output; // Shortcodes in WordPress MUST return content
    }


    private function minify_string($string) {
        // Split the string into an array of lines
        $lines = explode("\n", $string);

        // Trim whitespace from each line
        $trimmed_lines = array_map('trim', $lines);

        // Filter out empty lines
        $non_empty_lines = array_filter($trimmed_lines);

        // Join the lines back into a single string
        return implode(' ', $non_empty_lines);
    }

    public function openapp_qr_login($atts = array()){
        $sanitized_uri = filter_var($_SERVER['REQUEST_URI'], FILTER_SANITIZE_URL);
        $this->log_debug('openapp_qr_login triggered at ' . gmdate('Y-m-d H:i:s') . ' by ' . $sanitized_uri);

        if (is_user_logged_in()) {
           return '<p><strong>OA login:</strong> user already login <a href="' . wp_logout_url( home_url()) .'" title="Logout">'. __('Logout'). '</a></p>';
        }

        if ( is_admin() ) {
            return;
        }

        // parse the shortcode attributes
        $args = shortcode_atts(array(
            'lang' => sanitize_text_field(get_bloginfo('language')),
        ), $atts, 'openapp_qr_login');

        wp_enqueue_script('openappgw-js');

        $sessionId = $this->get_session_id();
        $token = $this->get_cart_id_by_session($sessionId);
        $nonce = wp_create_nonce('wp_rest');

        // Enqueue the script (only if it hasn't been enqueued already)
        if(!wp_script_is('openappgw-js-3', 'enqueued')) {
            wp_enqueue_script('openappgw-js-3', OPENAPPGW_PLUGIN_DIR_URL . 'assets/js/openapp-shortcodes-2.js', array('jquery'), OPENAPPGW_WOOCOMMERCE_GATEWAY, true);

            wp_enqueue_style('openapp-css', OPENAPPGW_PLUGIN_DIR_URL . 'assets/css/openapp-shortcode.css', array(), OPENAPPGW_WOOCOMMERCE_GATEWAY);

            // Pass PHP variables to JavaScript
            $localize_data = array(
                'baseUrl' => esc_url($this->get_base_url()),
                'qrCodeUrl' => esc_url_raw( rest_url( 'openapp/v1/qr_code' ) ),
                'oaRedirectionUrl' => esc_url_raw( rest_url( 'openapp/v1/oa_redirection' ) ),
                'oaLoginUrl' => esc_url_raw( rest_url( 'openapp/v1/oa_login' ) ),
                'cartId' => sanitize_text_field($token),
                'errorTextMessage' => esc_html__('OpenApp QR Login. Something went wrong.', 'openapp-gateway-for-woocommerce')
            );
            wp_localize_script('openappgw-js-3', 'openappVars2', $localize_data);
        }

        $sessionActive = $this->user_session_is_active($token);

        if(!$sessionActive || is_null($token)){
            $activateLoginButtonHtml = '';

            $loginIcon = '<svg xmlns="http://www.w3.org/2000/svg" class="icon icon-tabler icon-tabler-login" width="24" height="24" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
  <path stroke="none" d="M0 0h24v24H0z" fill="none"/>
  <path d="M14 8v-2a2 2 0 0 0 -2 -2h-7a2 2 0 0 0 -2 2v12a2 2 0 0 0 2 2h7a2 2 0 0 0 2 -2v-2" />
  <path d="M20 12h-13l3 -3m0 6l-3 -3" />
</svg>';

            $activateLoginButtonHtml .= '<p><a href="/?oa-set-guest-session=1" class="OpenAppButton" id="jsOaCheckSession2">' . __('Login via OpenApp') . $loginIcon . '</a></p>';

            return $activateLoginButtonHtml;
        }

        ob_start(); // Start capturing output into a buffer

        $output = '';

        $output .= '<div class="OpenAppCheckout-loading OpenAppCheckoutLogin" data-letter="I" data-version="1" data-merchantId="" data-integrationProfileId="" data-token="" data-lang="'. $args['lang'] .'"></div>';

        ob_end_clean(); // End capture and discard buffer
        return $output; // Shortcodes in WordPress MUST return content
    }

}
