<?php
require_once dirname(__FILE__) . '/../OpenappAbstractRESTController.php';

use Symfony\Component\HttpFoundation\Request;

/**
 * This REST endpoint returns product catalog
 * Mirrors WooCommerce /products endpoint functionality
 */
class ps_openappproductsModuleFrontController extends OpenappAbstractRESTController
{
    /**
     * Default pagination values
     */
    private $defaultPage = 1;
    private $defaultPerPage = 20;
    private $maxPerPage = 100;

    protected function processGetRequest()
    {
        header('Content-Type: application/json; charset=utf-8');

        $request = Request::createFromGlobals();
        $headers = $request->headers->all();

        $context = 'openapp_products';
        $this->ct_custom_log("Products endpoint called", $context);

        // Verify HMAC
        $validHmac = $this->isRequestValid($headers);

        if (!$validHmac) {
            $this->sendError('invalid_auth', $this->trans('Unauthorized request', [], 'Modules.Openapp.Products'), 403);
        }

        // Check if pagination params are explicitly provided
        $usePagination = isset($_GET['page']) || isset($_GET['per_page']);

        // Get pagination parameters (only used if pagination is requested)
        $page = $usePagination ? max(1, (int)($_GET['page'] ?? 1)) : 1;
        $perPage = $usePagination ? min($this->maxPerPage, max(1, (int)($_GET['per_page'] ?? $this->defaultPerPage))) : null;
        $fullMode = isset($_GET['full']) && strtolower($_GET['full']) === 'yes';

        $this->ct_custom_log("Page: $page, Per Page: " . ($perPage ?? 'all') . ", Full Mode: " . ($fullMode ? 'yes' : 'no') . ", Use Pagination: " . ($usePagination ? 'yes' : 'no'), $context);

        if ($usePagination) {
            // Get total count for pagination
            $totalItems = $this->getTotalProductCount();
            $totalPages = (int)ceil($totalItems / $perPage);

            // Get products with pagination
            $products = $this->getProducts($page, $perPage, $fullMode);

            $this->ct_custom_log("Products count: " . count($products) . ", Total: $totalItems", $context);

            // Build paginated response
            $response = [
                'data' => $products,
                'pagination' => [
                    'page' => $page,
                    'per_page' => $perPage,
                    'total_items' => $totalItems,
                    'total_pages' => $totalPages,
                    'has_next_page' => $page < $totalPages,
                    'has_prev_page' => $page > 1
                ]
            ];
        } else {
            // Return flat array (no pagination)
            $response = $this->getAllProducts($fullMode);
            $this->ct_custom_log("Products count: " . count($response), $context);
        }

        // Calculate X-Server-Authorization and set the header
        $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);

        if ($expectedXServerAuth !== null) {
            $this->ct_custom_log("X-Server-Authorization: " . $expectedXServerAuth, $context);
            header('X-Server-Authorization: ' . $expectedXServerAuth);
        }

        $this->ajaxRender(json_encode($response));
        die;
    }

    /**
     * Get total count of active products including variations
     *
     * @return int
     */
    private function getTotalProductCount()
    {
        $shopId = (int)Context::getContext()->shop->id;
        $langId = (int)Configuration::get('PS_LANG_DEFAULT');

        // Get all active product IDs
        $sql = new DbQuery();
        $sql->select('p.id_product');
        $sql->from('product', 'p');
        $sql->innerJoin('product_shop', 'ps', 'p.id_product = ps.id_product AND ps.id_shop = ' . $shopId);
        $sql->where('ps.active = 1');

        $productIds = Db::getInstance()->executeS($sql);

        $total = 0;
        foreach ($productIds as $row) {
            $productId = (int)$row['id_product'];
            $product = new Product($productId, false, $langId);

            if (!Validate::isLoadedObject($product)) {
                continue;
            }

            // Count variations; if none exist, count the base product
            $combinations = $product->getAttributeCombinations($langId);
            if (!empty($combinations)) {
                $uniqueCombinations = [];
                foreach ($combinations as $combination) {
                    $combinationId = (int)$combination['id_product_attribute'];
                    $uniqueCombinations[$combinationId] = true;
                }
                $total += count($uniqueCombinations);
            } else {
                $total++;
            }
        }

        return $total;
    }

    /**
     * Get all products without pagination (flat array)
     *
     * @param bool $fullMode Include extended product data
     * @return array
     */
    private function getAllProducts($fullMode)
    {
        $langId = (int)Configuration::get('PS_LANG_DEFAULT');
        $shopId = (int)Context::getContext()->shop->id;

        // Get all active product IDs
        $sql = new DbQuery();
        $sql->select('p.id_product');
        $sql->from('product', 'p');
        $sql->innerJoin('product_shop', 'ps', 'p.id_product = ps.id_product AND ps.id_shop = ' . $shopId);
        $sql->where('ps.active = 1');
        $sql->orderBy('p.id_product ASC');

        $productIds = Db::getInstance()->executeS($sql);

        $products = [];

        foreach ($productIds as $row) {
            $productId = (int)$row['id_product'];
            $product = new Product($productId, true, $langId);

            if (!Validate::isLoadedObject($product)) {
                continue;
            }

            // If product has variations, include them as separate entries; otherwise include base product
            $combinations = $product->getAttributeCombinations($langId);
            if (!empty($combinations)) {
                $processedCombinations = [];
                foreach ($combinations as $combination) {
                    $combinationId = (int)$combination['id_product_attribute'];

                    // Skip if already processed
                    if (isset($processedCombinations[$combinationId])) {
                        continue;
                    }
                    $processedCombinations[$combinationId] = true;

                    $products[] = $this->buildVariationData($product, $combination, $fullMode, $langId);
                }
            } else {
                $products[] = $this->buildProductData($product, $fullMode, $langId);
            }
        }

        return $products;
    }

    /**
     * Get products with pagination (counts variations individually)
     *
     * @param int $page Page number
     * @param int $perPage Products per page
     * @param bool $fullMode Include extended product data
     * @return array
     */
    private function getProducts($page, $perPage, $fullMode)
    {
        $langId = (int)Configuration::get('PS_LANG_DEFAULT');
        $shopId = (int)Context::getContext()->shop->id;

        // Calculate offset
        $offset = ($page - 1) * $perPage;

        // Get all active product IDs (we'll handle pagination manually due to variations)
        $sql = new DbQuery();
        $sql->select('p.id_product');
        $sql->from('product', 'p');
        $sql->innerJoin('product_shop', 'ps', 'p.id_product = ps.id_product AND ps.id_shop = ' . $shopId);
        $sql->where('ps.active = 1');
        $sql->orderBy('p.id_product ASC');

        $productIds = Db::getInstance()->executeS($sql);

        $products = [];
        $currentIndex = 0;
        $itemsAdded = 0;

        foreach ($productIds as $row) {
            if ($itemsAdded >= $perPage) {
                break;
            }

            $productId = (int)$row['id_product'];
            $product = new Product($productId, true, $langId);

            if (!Validate::isLoadedObject($product)) {
                continue;
            }

            // Check if we should include the base product
            if ($currentIndex >= $offset && $itemsAdded < $perPage) {
                $productData = $this->buildProductData($product, $fullMode, $langId);
                $products[] = $productData;
                $itemsAdded++;
            }
            $currentIndex++;

            // If product has variations, include them as separate entries
            $combinations = $product->getAttributeCombinations($langId);
            if (!empty($combinations)) {
                $processedCombinations = [];
                foreach ($combinations as $combination) {
                    if ($itemsAdded >= $perPage) {
                        break;
                    }

                    $combinationId = (int)$combination['id_product_attribute'];

                    // Skip if already processed
                    if (isset($processedCombinations[$combinationId])) {
                        continue;
                    }
                    $processedCombinations[$combinationId] = true;

                    if ($currentIndex >= $offset && $itemsAdded < $perPage) {
                        $variationData = $this->buildVariationData($product, $combination, $fullMode, $langId);
                        $products[] = $variationData;
                        $itemsAdded++;
                    }
                    $currentIndex++;
                }
            }
        }

        return $products;
    }

    /**
     * Build product data array
     *
     * @param Product $product
     * @param bool $fullMode
     * @param int $langId
     * @return array
     */
    private function buildProductData($product, $fullMode, $langId)
    {
        $productId = (int)$product->id;

        // Get current price (with reductions/sale prices applied)
        $price = Product::getPriceStatic($productId, true, null, 6, null, false, true);
        // Get regular price (without reductions) - 7th param $usereduc=false
        $regularPrice = Product::getPriceStatic($productId, true, null, 6, null, false, false);
        // Sale price is the discounted price if different from regular
        $salePrice = ($price < $regularPrice) ? $price : null;

        // Get stock info
        $stockQuantity = StockAvailable::getQuantityAvailableByProduct($productId);
        $stockStatus = $this->getStockStatus($product, $stockQuantity);

        $data = [
            'id' => $this->formatProductId($productId, 0),
            'sku' => $product->reference ?: '',
            'ean' => $product->ean13 ?: null,
            'name' => $product->name,
            'price' => $this->groszeFormat($price),
            'regular_price' => $this->groszeFormat($regularPrice),
            'sale_price' => $salePrice !== null ? $this->groszeFormat($salePrice) : 0,
            'stock_status' => $stockStatus,
            'stock_quantity' => $stockQuantity > 0 ? $stockQuantity : null
        ];

        // Add extended data in full mode
        if ($fullMode) {
            $data = array_merge($data, $this->getExtendedProductData($product, $langId));
        }

        return $data;
    }

    /**
     * Build variation/combination data array
     *
     * @param Product $product
     * @param array $combination
     * @param bool $fullMode
     * @param int $langId
     * @return array
     */
    private function buildVariationData($product, $combination, $fullMode, $langId)
    {
        $productId = (int)$product->id;
        $combinationId = (int)$combination['id_product_attribute'];

        // Get combination-specific data
        $combinationObj = new Combination($combinationId);

        // Build variation name with attributes
        $attributes = $combinationObj->getAttributesName($langId);
        $attributesString = implode(', ', array_column($attributes, 'name'));
        $variationName = $product->name . ' - ' . $attributesString;

        // Get current price for combination (with reductions/sale prices)
        $price = Product::getPriceStatic($productId, true, $combinationId, 6, null, false, true);
        // Get regular price for combination (without reductions) - 7th param $usereduc=false
        $regularPrice = Product::getPriceStatic($productId, true, $combinationId, 6, null, false, false);
        // Sale price is the discounted price if different from regular
        $salePrice = ($price < $regularPrice) ? $price : null;

        // Get stock for combination
        $stockQuantity = StockAvailable::getQuantityAvailableByProduct($productId, $combinationId);
        $stockStatus = $this->getStockStatus($product, $stockQuantity);

        // Use combination EAN if available, otherwise fall back to product EAN
        $ean = $combinationObj->ean13 ?: $product->ean13 ?: null;
        $sku = $combinationObj->reference ?: $product->reference ?: '';

        $data = [
            'id' => $this->formatProductId($productId, $combinationId),
            'sku' => $sku,
            'ean' => $ean,
            'name' => $variationName,
            'price' => $this->groszeFormat($price),
            'regular_price' => $this->groszeFormat($regularPrice),
            'sale_price' => $salePrice !== null ? $this->groszeFormat($salePrice) : 0,
            'stock_status' => $stockStatus,
            'stock_quantity' => $stockQuantity > 0 ? $stockQuantity : null
        ];

        // Add extended data in full mode
        if ($fullMode) {
            $data = array_merge($data, $this->getExtendedProductData($product, $langId, $combinationId));
        }

        return $data;
    }

    /**
     * Get extended product data for full mode
     *
     * @param Product $product
     * @param int $langId
     * @param int|null $combinationId
     * @return array
     */
    private function getExtendedProductData($product, $langId, $combinationId = null)
    {
        $productId = (int)$product->id;

        // Get images
        $images = $this->getProductImages($product, $langId, $combinationId);

        // Get categories
        $categories = $this->getProductCategories($product, $langId);

        // Get tags
        $tags = $this->getProductTags($product, $langId);

        // Get weight (use parent product weight for variations)
        $weight = $product->weight;

        // Get product type
        $type = 'simple';
        if ($product->getAttributeCombinations($langId)) {
            $type = $combinationId ? 'variation' : 'variable';
        }

        return [
            'type' => $type,
            'description' => strip_tags($product->description ?? ''),
            'short_description' => strip_tags($product->description_short ?? ''),
            'weight' => $this->formatMeasurement($weight),
            'weight_unit' => Configuration::get('PS_WEIGHT_UNIT'),
            'length' => $this->formatMeasurement($product->depth),
            'width' => $this->formatMeasurement($product->width),
            'height' => $this->formatMeasurement($product->height),
            'dimension_unit' => Configuration::get('PS_DIMENSION_UNIT'),
            'permalink' => $this->context->link->getProductLink(
                $product,
                null,
                null,
                null,
                $langId,
                null,
                $combinationId ?: 0
            ),
            'images' => $images,
            'categories' => $categories,
            'tags' => $tags,
            'attributes' => $combinationId ? $this->getCombinationAttributes($combinationId, $langId) : null
        ];
    }

    /**
     * Get product images
     *
     * @param Product $product
     * @param int $langId
     * @param int|null $combinationId
     * @return array
     */
    private function getProductImages($product, $langId, $combinationId = null)
    {
        $images = [];
        $productId = (int)$product->id;

        // Try to get combination-specific images first
        if ($combinationId) {
            $combinationImages = $product->getCombinationImages($langId);
            if (isset($combinationImages[$combinationId])) {
                foreach ($combinationImages[$combinationId] as $img) {
                    $images[] = $this->context->link->getImageLink(
                        $product->link_rewrite,
                        $img['id_image'],
                        'home_default'
                    );
                }
            }
        }

        // If no combination images, get all product images
        if (empty($images)) {
            $productImages = $product->getImages($langId);
            foreach ($productImages as $img) {
                $images[] = $this->context->link->getImageLink(
                    $product->link_rewrite,
                    $img['id_image'],
                    'home_default'
                );
            }
        }

        return $images;
    }

    /**
     * Get product categories
     *
     * @param Product $product
     * @param int $langId
     * @return array
     */
    private function getProductCategories($product, $langId)
    {
        $categories = [];
        $productCategories = $product->getCategories();

        foreach ($productCategories as $categoryId) {
            $category = new Category($categoryId, $langId);
            if (Validate::isLoadedObject($category) && $category->active) {
                $categories[] = $category->name;
            }
        }

        return $categories;
    }

    /**
     * Get product tags
     *
     * @param Product $product
     * @param int $langId
     * @return array
     */
    private function getProductTags($product, $langId)
    {
        $tags = [];
        $productTags = $product->getTags($langId);

        if (is_string($productTags) && !empty($productTags)) {
            $tags = explode(', ', $productTags);
        }

        return $tags;
    }

    /**
     * Get combination attributes
     *
     * @param int $combinationId
     * @param int $langId
     * @return array
     */
    private function getCombinationAttributes($combinationId, $langId)
    {
        // Use PrestaShop DbQuery ORM builder for cleaner queries
        // Note: Combination::getAttributesName() only returns attribute values without group names
        $sql = new DbQuery();
        $sql->select('agl.name as group_name, al.name as attr_name');
        $sql->from('product_attribute_combination', 'pac');
        $sql->innerJoin('attribute', 'a', 'a.id_attribute = pac.id_attribute');
        $sql->innerJoin('attribute_lang', 'al', 'al.id_attribute = a.id_attribute AND al.id_lang = ' . (int)$langId);
        $sql->innerJoin('attribute_group_lang', 'agl', 'agl.id_attribute_group = a.id_attribute_group AND agl.id_lang = ' . (int)$langId);
        $sql->where('pac.id_product_attribute = ' . (int)$combinationId);

        $rows = Db::getInstance()->executeS($sql);

        $result = [];
        if ($rows) {
            foreach ($rows as $row) {
                $result[strtolower($row['group_name'])] = $row['attr_name'];
            }
        }

        return $result;
    }

    /**
     * Get stock status string
     *
     * @param Product $product
     * @param int $stockQuantity
     * @return string
     */
    private function getStockStatus($product, $stockQuantity)
    {
        if ($stockQuantity > 0) {
            return 'instock';
        }

        // Check if backorders are allowed
        if ($product->out_of_stock == 1) {
            return 'onbackorder';
        }

        return 'outofstock';
    }
}
