<?php
/**
 * PSOPENAPP
 * 2007-2023 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 *  @author    PrestaShop SA <contact@prestashop.com>
 *  @copyright 2007-2023 PrestaShop SA
 *  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

use PrestaShop\PrestaShop\Adapter\Presenter\Cart\CartPresenter;

require_once(__DIR__ . '/classes/OpenappAPIRoutes.php');
require_once(__DIR__ . '/classes/OpenappApiClientTrait.php');

class Ps_openapp extends PaymentModule
{
    use OpenappApiClientTrait;

    private $shippingMethodOptions;

    public function __construct()
    {
        $this->name = 'ps_openapp';
        $this->tab = 'payments_gateways';
        $this->version = '2.0.0';
        $this->author = 'CreateIT';
        $this->need_instance = 0;

        /**
         * Set $this->bootstrap to true if your module is compliant with bootstrap (PrestaShop 1.6)
         */
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('PS Openapp');
        $this->description = $this->l('OpenApp gateway for Prestashop');
        $this->ps_versions_compliancy = array('min' => '1.7', 'max' => _PS_VERSION_);

        $this->shippingMethodOptions = array(
            array('id_option' => '',           'name' => 'Disabled'),
            array('id_option' => 'INPOST_APM', 'name' => 'InPost Paczkomat'),
            array('id_option' => 'ORLEN_APM', 'name' => 'Delivery with Orlen Paczka'),
            array('id_option' => 'POCZTA_POLSKA_APM', 'name' => 'Delivery with Poczta Polska pickup (machines, pickup points)'),
            array('id_option' => 'DHL_PICKUP', 'name' => 'Pickup from DHL pickup point'),
            array('id_option' => 'DPD_PICKUP', 'name' => 'Pickup from DPD pickup point'),
            array('id_option' => 'INSTORE_PICKUP', 'name' => 'Self-pickup by customer in your store'),
            array('id_option' => 'INPOST_COURIER', 'name' => 'Delivery with InPost courier'),
            array('id_option' => 'DHL_COURIER', 'name' => 'Delivery using DHL courier'),
            array('id_option' => 'DPD_COURIER', 'name' => 'Delivery with DPD courier'),
            array('id_option' => 'UPS_COURIER', 'name' => 'Delivery with UPS courier'),
            array('id_option' => 'FEDEX_COURIER', 'name' => 'Delivery with FEDEX courier'),
            array('id_option' => 'GLS_COURIER', 'name' => 'Delivery with GLS courier'),
            array('id_option' => 'POCZTEX_COURIER', 'name' => 'Delivery with POCZTEX courier'),
            array('id_option' => 'GEIS_COURIER', 'name' => 'Delivery with GEIS courier'),
            array('id_option' => 'ELECTRONIC', 'name' => 'Electronic delivery, eg. license, cinema tickets'),
        );

    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        if (extension_loaded('curl') == false)
        {
            $this->_errors[] = $this->l('You have to enable the cURL extension on your server to install this module');
            return false;
        }

        include(dirname(__FILE__) . '/sql/install.php');
        include(dirname(__FILE__) . '/sql/install_oa_persistent_cart.php');

        $this->installReturnStates();

        return parent::install()
            && $this->registerHook('displayHeader')
            && $this->registerHook('displayShoppingCart')
            && $this->registerHook('displayCheckoutSummaryTop')

            && $this->registerHook('displayCustomerLoginFormAfter')

            && $this->registerHook('actionDispatcherBefore')
            && $this->registerHook('actionCartSave')

            && $this->registerHook('actionOrderStatusUpdate')

            && $this->registerHook('openapp_qr_order')
            && $this->registerHook('openapp_qr_login')

            && $this->registerHook('displayAdminOrderMain')


            && $this->registerHook('actionCarrierUpdate');
    }

    public function uninstall()
    {
        include(dirname(__FILE__) . '/sql/uninstall.php');
        return parent::uninstall();
    }

    public function installReturnStates()
    {
        // Keep reference to default "Waiting for confirmation" as received state
        if (!Configuration::get('OA_RETURN_STATE_RECEIVED')) {
            Configuration::updateValue('OA_RETURN_STATE_RECEIVED', 1);
        }

        if (!Configuration::get('OA_RETURN_STATE_ACCEPTED')) {
            $accepted = $this->createReturnState($this->l('OA Accepted'), '#27ae60');
            Configuration::updateValue('OA_RETURN_STATE_ACCEPTED', $accepted);
        }

        if (!Configuration::get('OA_RETURN_STATE_REJECTED')) {
            $rejected = $this->createReturnState($this->l('OA Rejected'), '#e74c3c');
            Configuration::updateValue('OA_RETURN_STATE_REJECTED', $rejected);
        }

    }

    private function createReturnState($label, $color)
    {
        $state = new OrderReturnState();
        $state->color = $color;
        $state->active = 1;
        $state->name = array();

        foreach (Language::getLanguages(false) as $lang) {
            $state->name[$lang['id_lang']] = $label;
        }

        $state->save();

        return (int) $state->id;
    }


    /**
     * Load the configuration form
     */
    public function getContent()
    {
        $output = '';
        if (((bool)Tools::isSubmit('submitPs_openappModule')) == true) {
            $this->postProcess();
            $output .= '<div class="bootstrap"><div class="module_confirmation conf confirm alert alert-success"><button type="button" class="close" data-dismiss="alert">×</button>'.$this->trans('The settings have been updated.', [], 'Admin.Notifications.Success').'</div></div>';
        }
        $this->context->smarty->assign('module_dir', $this->_path);

        return $output.$this->renderForm();
    }

    /**
     * Create the form that will be displayed in the configuration of your module.
     */
    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitPs_openappModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(), /* Add values for your inputs */
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($this->getConfigForm()));
    }

    /**
     * Create the structure of your form.
     */
    protected function getConfigForm()
    {
        $form = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Settings'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'text',
                        'label' => $this->trans('API Key', [], 'Modules.Openapp.Admin'),
                        'name' => 'OA_API_KEY',
                        'class' => 'fixed-width-xxl',
                        'desc' => $this->trans('Your Openapp API Key.', [], 'Modules.Openapp.Admin'),
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->trans('Secret', [], 'Modules.Openapp.Admin'),
                        'name' => 'OA_API_SECRET',
                        'class' => 'fixed-width-xxl',
                        'desc' => $this->trans('Your Openapp Secret.', [], 'Modules.Openapp.Admin'),
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->trans('Merchant ID', [], 'Modules.Openapp.Admin'),
                        'name' => 'OA_MERCHANT_ID',
                        'class' => 'fixed-width-xxl',
                        'desc' => $this->trans('OpenApp Merchant ID' , [], 'Modules.Openapp.Admin'),
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->trans('Profile ID', [], 'Modules.Openapp.Admin'),
                        'name' => 'OA_PROFILE_ID',
                        'class' => 'fixed-width-xxl',
                        'desc' => $this->trans('OpenApp Profile ID', [], 'Modules.Openapp.Admin'),
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->trans('API Base Url', [], 'Modules.Openapp.Admin'),
                        'name' => 'OA_API_BASE_URL',
                        'class' => '',
                        'desc' => $this->trans('API Base Url (https://api.openapp.com)', [], 'Modules.Openapp.Admin'),
                    ],
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Disable Validation'),
                        'name' => 'OA_DISABLE_VALIDATION_MODE',
                        'is_bool' => true,
                        'desc' => $this->trans('If checked, the plugin will disable validation of OpenApp incoming requests. Use only for testing purposes.', [], 'Modules.Openapp.Admin'),
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => true,
                                'label' => $this->l('Enabled')
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => false,
                                'label' => $this->l('Disabled')
                            )
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Enable OA login'),
                        'name' => 'OA_ENABLE_LOGIN',
                        'is_bool' => true,
                        'desc' => $this->trans('If checked, the OALogin functionality will be enabled', [], 'Modules.Openapp.Admin'),
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => true,
                                'label' => $this->l('Enabled')
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => false,
                                'label' => $this->l('Disabled')
                            )
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Enable debug log'),
                        'name' => 'OA_ENABLE_LOG',
                        'is_bool' => true,
                        'desc' => $this->trans('If checked, the plugin will log debug information to [prestashop_root]/var/logs/oadebug.log file.', [], 'Modules.Openapp.Admin'),
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => true,
                                'label' => $this->l('Enabled')
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => false,
                                'label' => $this->l('Disabled')
                            )
                        ),
                    ),
                ],
                'submit' => [
                    'title' => $this->l('Save changes'),
                ],
            ],
        ];


        $form['form']['input'][] = array(
            'type' => 'html',
            'name' => 'separator',
            'html_content' => '<hr style="margin-top: 20px; margin-bottom: 20px;">',
        );

        $carriers = Carrier::getCarriers($this->context->language->id, true, false, false, null, Carrier::ALL_CARRIERS);

        foreach ($carriers as $carrier) {
            $carrierInput = array(
                'type' => 'select',
                'label' => $this->trans('Mapping for Carrier: ', [], 'Modules.Openapp.Admin').$carrier['name'],
                'name' => 'CARRIER_MAP_OA_'.$carrier['id_carrier'],
                'options' => array(
                    'query' => $this->shippingMethodOptions,
                    'id' => 'id_option',
                    'name' => 'name'
                ),
            );
            // Add this input to your form inputs
            $form['form']['input'][] = $carrierInput;
        }



        return $form;
    }

    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
        $values = array(
            'OA_API_KEY' => Tools::getValue('OA_API_KEY', Configuration::get('OA_API_KEY')),
            'OA_API_SECRET' => Tools::getValue('OA_API_SECRET', Configuration::get('OA_API_SECRET')),
            'OA_MERCHANT_ID' => Tools::getValue('OA_MERCHANT_ID', Configuration::get('OA_MERCHANT_ID')),
            'OA_PROFILE_ID' => Tools::getValue('OA_PROFILE_ID', Configuration::get('OA_PROFILE_ID')),
            'OA_API_BASE_URL' => Tools::getValue('OA_API_BASE_URL', Configuration::get('OA_API_BASE_URL')),
            'OA_DISABLE_VALIDATION_MODE' => Configuration::get('OA_DISABLE_VALIDATION_MODE', false),
            'OA_ENABLE_LOGIN' => Configuration::get('OA_ENABLE_LOGIN', false),
            'OA_ENABLE_LOG' => Configuration::get('OA_ENABLE_LOG', false)
        );

        $carriers = Carrier::getCarriers($this->context->language->id, true, false, false, null, Carrier::ALL_CARRIERS);
        foreach ($carriers as $carrier) {
            $values['CARRIER_MAP_OA_'.$carrier['id_carrier']] = Tools::getValue('CARRIER_MAP_OA_'.$carrier['id_carrier'], Configuration::get('CARRIER_MAP_OA_'.$carrier['id_carrier']));
        }

        return $values;

    }

    /**
     * Save form data.
     */
    protected function postProcess()
    {
        if (Tools::isSubmit('submitPs_openappModule')) {


            $form_values = $this->getConfigFormValues();
            foreach (array_keys($form_values) as $key) {
                Configuration::updateValue($key, Tools::getValue($key));
            }

            $carriers = Carrier::getCarriers($this->context->language->id, true, false, false, null, Carrier::ALL_CARRIERS);
            foreach ($carriers as $carrier) {
                Configuration::updateValue('CARRIER_MAP_OA_'.$carrier['id_carrier'], Tools::getValue('CARRIER_MAP_OA_'.$carrier['id_carrier']));
            }

        }

    }

    /**
     * Add OpenApp script in header
     *
     * @param array Hook parameters
     *
     * @return array|null
     */
    public function hookDisplayHeader()
    {
        $this->context->controller->addJS($this->_path . 'views/js/openapp-shortcodes-1.js');
        $this->context->controller->addJS($this->_path . 'views/js/openapp-shortcodes-2.js');
        $this->context->controller->addCSS($this->_path . 'views/css/openapp-shortcode.css', 'all');

        // You can add your module stylesheets from here
        $this->context->controller->addCSS($this->_path . 'views/css/front.css', 'all');
        $this->context->smarty->assign([
            'openapp_header_script' => 'https://static.prd.open-pay.com/dist/barcode-library/openapp.min.1.0.0.js',
            'openappVars2_cartId' => '',
            'openappVars2_nonce' => '',
            'openappVars2_errorTextMessage' => $this->l('Cart is empty.')
        ]);

        if( !is_null( $this->context->cart->id ) )
        {
            $this->context->smarty->assign([
                'openappVars2_cartId' => $this->context->cart->id,
            ]);
        }
        return $this->display(__FILE__, 'openapp-header.tpl');
    }

    /**
     * Display the QR Code in cart page
     */
    public function hookDisplayShoppingCart($params)
    {
        if( !empty( $this->context->cart->id ) )
        {
            return $this->display(__FILE__, 'openapp-qr1.tpl');
        }
    }

    /**
     * Display the QR Code in checkout page
     */
    public function hookDisplayCheckoutSummaryTop()
    {

        if( !empty( $this->context->cart->id ) )
        {
            return $this->display(__FILE__, 'openapp-qr1.tpl');
        }
    }

    /**
     * Display OaLogin QR code on login page
     */
    public function hookDisplayCustomerLoginFormAfter($params)
    {
        return $this->display(__FILE__, 'openapp-qr2.tpl');
    }

    public function hookDisplayAdminOrderMain($params)
    {
        $idOrder = isset($params['id_order']) ? (int) $params['id_order'] : 0;
        if (!$idOrder) {
            return '';
        }

        $order = new Order($idOrder);
        if (!Validate::isLoadedObject($order)) {
            return '';
        }

        $returns = $this->getOpenAppReturnsForOrder($idOrder, $order);
        if (!$returns || count($returns) === 0) {
            return '';
        }

        $this->context->smarty->assign(array(
            'openapp_returns' => $returns,
            'openapp_order' => $order,
            'openapp_admin_token' => Tools::getAdminTokenLite('AdminOrders'),
            'openapp_employee_id' => ($this->context->employee && $this->context->employee->id) ? (int) $this->context->employee->id : 0,
            'openapp_return_action_url' => $this->context->link->getModuleLink('ps_openapp', 'orderreturnaction'),
            'openapp_received_state' => (int) Configuration::get('OA_RETURN_STATE_RECEIVED', 1),
        ));

        return $this->display(__FILE__, 'views/templates/admin/openapp-returns.tpl');
    }


    private function getOpenAppReturnsForOrder($idOrder, $order = null)
    {
        $langId = (int) $this->context->language->id;
        $productNames = $this->buildOrderProductNameMap($idOrder, $order);

        $sql = new DbQuery();
        $sql->select('r.id_order_return, r.state, r.date_add, r.question, l.name AS state_name');
        $sql->from('order_return', 'r');
        $sql->leftJoin('order_return_state_lang', 'l', 'r.state = l.id_order_return_state AND l.id_lang = ' . $langId);
        $sql->where('r.id_order = ' . (int) $idOrder);
        $sql->where("r.question LIKE 'OA Return %'");

        $rows = Db::getInstance()->executeS($sql);
        if (!$rows) {
            return array();
        }

        $returns = array();
        foreach ($rows as $row) {
            $meta = $this->parseReturnQuestion($row['question']);
            $payload = null;
            if (!empty($meta['oaReturnId'])) {
                $payload = $this->getPayloadForReturn($idOrder, $meta['oaReturnId']);
                if (isset($payload['returnedProducts']) && is_array($payload['returnedProducts'])) {
                    foreach ($payload['returnedProducts'] as $idx => $product) {
                        $productIdRaw = isset($product['id']) ? $product['id'] : (isset($product['productId']) ? $product['productId'] : null);
                        if ($productIdRaw && isset($productNames[$productIdRaw])) {
                            $payload['returnedProducts'][$idx]['displayName'] = $productNames[$productIdRaw];
                        }
                    }
                }
            }

            $returns[] = array(
                'id_order_return' => (int) $row['id_order_return'],
                'state' => (int) $row['state'],
                'state_name' => $row['state_name'],
                'created_at' => $row['date_add'],
                'question' => $row['question'],
                'case_id' => isset($meta['caseId']) ? $meta['caseId'] : '',
                'oa_return_id' => isset($meta['oaReturnId']) ? $meta['oaReturnId'] : '',
                'return_type' => isset($meta['type']) ? $meta['type'] : '',
                'payload' => $payload,
                'admin_return_link' => $this->context->link->getAdminLink('AdminReturn')
                    . '&updateorder_return&id_order_return=' . (int) $row['id_order_return'],
            );
        }

        return $returns;
    }

    private function buildOrderProductNameMap($idOrder, $order = null)
    {
        $names = array();
        if (!$order || !Validate::isLoadedObject($order)) {
            $order = new Order((int) $idOrder);
        }
        if (!$order || !Validate::isLoadedObject($order)) {
            return $names;
        }

        $details = OrderDetail::getList((int) $order->id);
        foreach ($details as $detail) {
            $productId = (int) $detail['product_id'];
            $productAttributeId = (int) $detail['product_attribute_id'];
            $key = $productId . '_' . $productAttributeId;
            $names[$key] = $detail['product_name'];
        }

        return $names;
    }

    private function parseReturnQuestion($question)
    {
        $meta = array();
        if (preg_match('/caseId:\s*([^|]+)/', $question, $m)) {
            $meta['caseId'] = trim($m[1]);
        }
        if (preg_match('/oaReturnId:\s*([^|]+)/', $question, $m)) {
            $meta['oaReturnId'] = trim($m[1]);
        }
        if (preg_match('/type:\s*([^\|]+)/', $question, $m)) {
            $meta['type'] = trim($m[1]);
        }

        return $meta;
    }

    private function getPayloadForReturn($idOrder, $oaReturnId)
    {
        $messages = Message::getMessagesByOrderId($idOrder, true);

        foreach ($messages as $message) {
            $content = isset($message['message']) ? $message['message'] : '';
            if (strpos($content, 'OA_RETURN_PAYLOAD:') === 0) {
                $json = substr($content, strlen('OA_RETURN_PAYLOAD:'));
                $decoded = json_decode($json, true);
                if (!$decoded) {
                    continue;
                }

                if (isset($decoded['oaOrderReturnId']) && $decoded['oaOrderReturnId'] === $oaReturnId) {
                    return $decoded;
                }
            }
        }

        return null;
    }



    public function hookactionDispatcherBefore($controller)
    {
        if ($controller['controller_type'] === 1) {
            $urlParts = parse_url($_SERVER['REQUEST_URI']);
            $pathSegments = explode('/', trim($urlParts['path'], '/'));

            // Check if the URL is exactly 'openapp/qr_code'
            if (count($pathSegments) === 2 && $pathSegments[0] === 'openapp') {
                $module_name = 'ps_openapp';

                $controllers_mapping = array(
                    'qr_code'             => 'qrcode',
                    'oa_login'            => 'oalogin',
                    'oa_redirection'      => 'oaredirection',
                    'basket' => 'basket',
                    'basket_recalculate' => 'basket_recalculate',
                    'cart' => 'cart',
                    'order' => 'order',
                    'order-return' => 'orderreturn',
                    'order-return-action' => 'orderreturnaction',
                    'identity' => 'identity',
                    'products' => 'products',
                );

                $controller_name = isset($controllers_mapping[$pathSegments[1]]) ? $controllers_mapping[$pathSegments[1]] : null;

                if(! is_null($controller_name)){
                    $_GET['fc'] = 'module';
                    $_GET['module'] = $module_name;
                    $_GET['controller'] = $controller_name;

                    $module = Module::getInstanceByName($module_name);
                    $controller_class = 'PageNotFoundController';

                    if (Validate::isLoadedObject($module) && $module->active) {
                        $controllers = Dispatcher::getControllers(_PS_MODULE_DIR_ . "$module_name/controllers/front/");
                        if (isset($controllers[strtolower($controller_name)])) {
                            $controller_file = _PS_MODULE_DIR_ . "$module_name/controllers/front/{$controller_name}.php";
                            if (file_exists($controller_file)) {
                                include_once $controller_file;
                                $controller_class = $module_name . $controller_name . 'ModuleFrontController';
                            }
                        }
                    }

                    if ($controller_class === 'PageNotFoundController') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => true,
                            'message' => 'Openapp - This endpoint is not defined.',
                            'code' => 410
                        ]);
                        die;
                    }

                    $controller = Controller::getController($controller_class);
                    if ($controller) {
                        $controller->restRun();
                    } else {
                        error_log("Failed to initialize controller: $controller_class");
                    }
                }
            }
        }
    }


    public function hookOpenapp_QR_Order($params)
    {
        if( isset($params['lang']) )
        {
            $lang = ( $params['lang'] === 'default' ) ? $this->context->language->iso_code : $params['lang'];
            $this->context->smarty->assign([
                'lang' => $lang
            ]);
        }

        return $this->display(__FILE__, 'openapp-qr-order.tpl');
    }


    public function hookOpenapp_QR_Login($params)
    {

        // if enable login is true
        if(Configuration::get('OA_ENABLE_LOGIN', false)){

            if (isset($_GET['oa-set-guest-session']) && $_GET['oa-set-guest-session'] == 1) {


                if (!isset($this->context->cart) || !$this->context->cart->id) {
                    $cart = new Cart();
                    $cart->id_currency = (int)Configuration::get('PS_CURRENCY_DEFAULT'); // Get default currency from configuration
                    $cart->id_lang = (int)Configuration::get('PS_LANG_DEFAULT'); // Get default language from configuration
                    $cart->id_shop = (int)Configuration::get('PS_SHOP_DEFAULT'); // Get default shop from configuration
                    $cart->add();

                    $this->ct_custom_log("oa-login: new session created");

                    // Update the context with the new cart
                    $this->context->cart = $cart;
                    $this->context->cookie->id_cart = $cart->id;

                    // $this->hookActionCartSave(array('cart' => $this->context->cart));

                    Hook::exec('actionCartSave', array('cart' => $cart));

                    // Parse the current URL and its components
                    $urlComponents = parse_url($_SERVER['REQUEST_URI']);
                    $queryParams = array();
                    if (isset($urlComponents['query'])) {
                        parse_str($urlComponents['query'], $queryParams);
                    }

                    // Remove the oa-set-guest-session parameter
                    unset($queryParams['oa-set-guest-session']);

                    // Rebuild the query string without the oa-set-guest-session parameter
                    $newQueryString = http_build_query($queryParams);
                    $newUrl = $urlComponents['path'] . ($newQueryString ? '?' . $newQueryString : '');

                    // Redirect to the new URL
                    Tools::redirect($newUrl);
                    exit;
                }
            }

            if( isset($params['lang']) )
            {
                $lang = ( $params['lang'] === 'default' ) ? $this->context->language->iso_code : $params['lang'];
                $this->context->smarty->assign([
                    'lang' => $lang
                ]);
            }

            $is_logged = false;
            $sessionActive = false;

            if (isset($this->context->cart)) {
                $is_logged = $this->context->customer->isLogged();

                // Get cart details
                $cart = $this->context->cart;
                $sessionActive = (isset($cart->id) && $cart->id > 0) ? true : false;
            }

            $this->context->smarty->assign([
                'currentUrl' => $_SERVER['REQUEST_URI'],
                'is_logged' => $is_logged,
                'sessionActive' => $sessionActive,
            ]);

            return $this->display(__FILE__, 'openapp-qr-login.tpl');
        }

        return false;
    }


    /**
     * store_cart_in_db
     */
    public function hookActionCartSave($params) {

        $orderAgainAction =  isset($_REQUEST['submitReorder']) && $_REQUEST['submitReorder'] == '1';

        // important condition, otherwise there is error 500
        if (!Validate::isLoadedObject($this->context->cart) && !$orderAgainAction)
        {
            return false;
        }

        if (empty($_REQUEST)) {
            return;
        }

        if (isset($params['cart']) && Validate::isLoadedObject($params['cart'])) {
            $cart_id = $params['cart']->id;

            if (!$cart_id) {
                return;
            }

            $this->ct_custom_log("store_cart_in_db");

            $table_name = _DB_PREFIX_ . 'ps_oa_persistent_cart';

            $hashed_session_id = hash('md5', $cart_id);
            $cart_id_safe = pSQL($cart_id);

            $cart_data = array();

            try {
                $cart_data = $this->get_cart_data($cart_id);
            } catch (exception $e) {
                // error_log($e->getMessage());
            }

            $cart_data_json = pSQL(json_encode($cart_data));

            $cart_expiry = (int)(time() + 60 * 60 * 24 * 30);

            $sql_select = "SELECT * FROM `" . $table_name . "` WHERE `cart_id` = '$cart_id_safe' ";
            $results = Db::getInstance()->executeS($sql_select);

            if ($results && count($results) > 0) {
                $update_query = "UPDATE `" . $table_name . "` SET  `cart_expiry` = '$cart_expiry', `cart_contents` = '$cart_data_json'
                            WHERE `cart_id` = '$cart_id_safe' ";
                Db::getInstance()->Execute($update_query);
            } else {
                $insert_query = "INSERT INTO `" . $table_name . "` (`cart_id`, `cart_contents`, `cart_session_id`, `cart_expiry`, `last_update`, `order_count`, `order_key`, `oaOrderId`, `oa_auth_token`, `oa_last_login`)
                            VALUES ('$cart_id_safe', '$cart_data_json', '$hashed_session_id', '$cart_expiry', '', '', null, null, null, null)";
                Db::getInstance()->Execute($insert_query);
            }

        }
    }

    /**
     * @TODO - unify 2 instances of ct_custom_log into 1 method
     */
    private function ct_custom_log($message, $context = '')
    {
        if (Configuration::get('OA_ENABLE_LOG', false)) {
            // Configure the logger
            $logger = new \Monolog\Logger('oa1');
            $logPath = _PS_ROOT_DIR_.'/var/logs/oadebug.log'; // Define the log path
            $logStreamHandler = new \Monolog\Handler\StreamHandler($logPath, \Monolog\Logger::DEBUG);
            $logger->pushHandler($logStreamHandler);

            // Log the message (you can change the level from debug to info, notice, warning, etc.)
            $logger->debug($message);
        }

        return false;
    }

    /**
     * oa_status_changed
     */

    public function hookActionOrderStatusUpdate($params) {
        $id_order = $params['id_order'];
        $newOrderStatus = $params['newOrderStatus'];

        $order = new Order((int) $id_order);
        if (!Validate::isLoadedObject($order)) {
            return;
        }
        $currentOrderStatus = $order->current_state;

        if ($newOrderStatus->id == $currentOrderStatus) {
            return;
        }


        /**
         * OA statuses
         */
        $osPreparation = Configuration::get('PS_OS_PREPARATION'); // Preparation in progress
        $osShipped = Configuration::get('PS_OS_SHIPPING'); // Shipped
        $osDelivered = Configuration::get('PS_OS_DELIVERED'); // Delivered
        $osCanceled = Configuration::get('PS_OS_CANCELED'); // Canceled
        $osPaymentAccepted = Configuration::get('PS_OS_PAYMENT'); // Payment accepted

        // Mapping Prestashop statuses to the ones expected by OpenApp
        $status_mapping = array(
            $osPaymentAccepted => 'ORDERED', // Assuming that 'Payment Accepted' means the order is placed
            $osPreparation => 'FULFILLED', // Assuming that 'Preparation in progress' means the order is fulfilled and ready for shipping or pickup
            $osShipped => 'SHIPPED',
            $osDelivered => 'DELIVERED',
            $osCanceled => 'CANCELLED_MERCHANT',
            // 'READY_FOR_PICKUP' might not have a direct equivalent in default PrestaShop statuses.
        );

        $this->ct_custom_log(var_export($status_mapping, true));

        // Check if the changed status exists in our mapping
        if (!isset($status_mapping[$newOrderStatus->id])) {
            return; // Exit if we don't need to send data for this status
        }

        /**
         * Get transactionId
         */
        $order = new Order((int) $id_order);
        $order_reference = $order->reference;
        $payments = $order->getOrderPaymentCollection();
        $oaOrderId = null;

        if ($payments && count($payments) > 0) {
            // Get the transaction_id from the first payment
            $payment = $payments[0]; // Get the first payment
            $oaOrderId = $payment->transaction_id;
        }


        $json_post_data1 = json_encode($oaOrderId, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $this->ct_custom_log($json_post_data1);

        if (!$oaOrderId) {
            return;
        }

        $carrier_id = $order->id_carrier;

        $operator = Configuration::get('CARRIER_MAP_OA_' . $carrier_id);
        if (!$operator) {
            return;
        }

        $order_data = [
            'oaOrderId' => $oaOrderId,
            'shopOrderId' => $order_reference,
            'shipments' => [
                [
                    'shipmentId' => $order_reference ."_". $operator, // Assuming the OA order ID serves as the shipment ID
                    'status' => $status_mapping[$newOrderStatus->id],
                    'notes' => '',
                    'operator' => $operator,
                ],
            ],
        ];

        $apiResponse = $this->sendOpenAppRequest('/v1/orders/multiFulfillment', 'POST', $order_data, 'openapp_basket_fulfillment');
        if (!$apiResponse['success']) {
            $this->ct_custom_log("OA fulfillment sync failed: " . $apiResponse['error']);
        }
    }


    private function get_cart_data($cart_id) {
        $cart = new Cart($cart_id);

        if(empty($cart->id)){
            return array();
        }

        $products = array();
        try {
            $products = $cart->getProducts(true);
        }
        catch (exception $e) {
            // $this->ct_custom_log($e->getMessage());
        }

        $cart_data = array();

        foreach ($products as $product) {
            // Check if there is a product attribute
            $productId = (string) $product['id_product'];
            $productIdAttribute = (string) isset($product['id_product_attribute']) ? $product['id_product_attribute'] : 0;

            $product_data = array(
                'id' => $productId,
                'id_product_attribute' => $productIdAttribute,
                'quantity' => $product['cart_quantity']
            );

            $cart_data[] = $product_data;
        }

        // Add customer and guest IDs
        $cart_contents['customer_id'] = $cart->id_customer; // Customer ID
        $cart_contents['guest_id'] = $cart->id_guest; // Guest ID (if applicable)

        $cart_contents['cart_contents']['products']  = $cart_data;

        return $cart_contents;
    }


    public function hookActionCarrierUpdate($params)
    {
        // Retrieve the old and new carrier IDs
        $id_carrier_old = (int) $params['id_carrier'];
        $id_carrier_new = (int) $params['carrier']->id;

        // Check if the old carrier ID has a custom configuration and update it for the new ID
        $oldConfigValue = Configuration::get('CARRIER_MAP_OA_' . $id_carrier_old);
        if ($oldConfigValue !== false) {
            Configuration::updateValue('CARRIER_MAP_OA_' . $id_carrier_new, $oldConfigValue);
        }
    }

    /**
     * Other
     */

    public function encodeJsonResponse($response)
    {

        return json_encode($response, JSON_UNESCAPED_SLASHES);
    }

}
